/* ------------------------------------------------------------------------------
  File: main.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Entry point for UM6 firmware
------------------------------------------------------------------------------ */ 

// Includes 
#include <math.h>
#include "stm32f10x.h"
#include "UM6_startup.h"
#include "UM6_config.h"
#include "UM6_states.h"
#include "UM6_usart.h"
#include "CHR_packet_handler.h"
#include "CHR_i2c.h"
#include "CHR_HMC.h"
#include "CHR_ITG3200.h"
#include "CHR_LSM303.h"
#include "CHR_matrix.h"
#include "CHR_delays.h"

extern uint8_t gClockInUse;

// Flags to indicate when specific sensors have new data ready
__IO uint8_t gGyroReady;
__IO uint8_t gAccelReady;
__IO uint8_t gMagReady;

__IO uint8_t gSendStateData;

__IO uint8_t gMagStatesReceived;
__IO uint8_t gAccelStatesReceived;
__IO uint8_t gGyroStatesReceived;

/*******************************************************************************
* Function Name  : main
* Input          : None
* Output         : None
* Return         : None
* Description    : Entry point for CHR-6dm firmware
*******************************************************************************/
int main(void)
{
	 uint32_t i;
	 uint32_t j;
	 int32_t new_data;
	 int32_t temp = 0;
	 int32_t returnval;
	 
	 uint8_t ITG_status;
	 uint8_t HMC_status;
	 uint8_t LSM_status;
	 
	 int32_t states_initialized = 0;
	 	 
	 gSensorData.new_accel_data = 0;
	 gSensorData.new_mag_data = 0;
	 
	 gGyrosCalibrated = 0;
	 gZeroGyroEnable = 0;
	 
	 // Initialize the IMU clocks, ADC, DMA controller, GPIO pins, etc.
	 Initialize_UM6();
	 
	 // Clear global data array
	 ClearGlobalData();
	 
	 // Fill gConfig structure from flash, or use default values if flash has not been initialized.
    GetConfiguration();

	 // Based on configuration settings, set serial baud rate
	 UpdateSerialBaud();
	 
	 gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] = 0;
	 
	 // Set i2c SDA ans SCl lines
	 i2cReadSDA();
	 i2cReadSCL();
	 
	 // Wait a while for supply voltages to stabilize
	 DelayMs( 50 );
	 	
	 // Initialize ITG sensor (rate gyros)
	 if( initializeITG( &ITG_status ) == 0 )
	 {
		  gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] |= UM6_GYRO_INIT_FAILED;
	 }
	 
	 
	 // Initialize HMC5843 (magnetic sensor)
	 if( initializeHMC( &HMC_status ) == 0 )
	 {
		  gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] |= UM6_MAG_INIT_FAILED;
	 }
	 
	 
	 // Intialize LSM (accels)
	 if( initializeLSM( &LSM_status ) == 0 )
	 {
		  gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] |= UM6_ACCEL_INIT_FAILED;
	 }
	 
	 // If the status register is non-zero, then there was an error during startup.  Send status register contents.
	 if( gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] != 0 )
	 {
		  SendStatusPacket();
	 }
	 
	 // Start "zero gyros" command.  When finished, the global flag gGyrosCalibrated will be set, and
	 // the main execution code will know to start estimating states and transmitting data.
	 if( gConfig.r[UM6_MISC_CONFIG] & UM6_GYRO_STARTUP_CAL )
	 {
		  StartGyroCalibration();
	 }
	 else
	 {
		  gGyrosCalibrated = 1;		  
	 }
	
	 // If in broadcast mode, start transmitting
	 if( gConfig.r[UM6_COMMUNICATION] & UM6_BROADCAST_ENABLED )
	 {
		  EnableBroadcastMode( (uint8_t)(gConfig.r[UM6_COMMUNICATION] & UM6_SERIAL_RATE_MASK) );
	 }
	 
	 // Start timer3 for prediction period tracking
	 TIM_SetCounter(TIM3,0);
	 TIM_Cmd(TIM3, ENABLE);
	 
	 // Start timer4 for sensor timeout detection
	 TIM_SetCounter(TIM4,0);
	 TIM_Cmd(TIM4, ENABLE);

	 // Configure and enable EXTI line for rate gyro.
	 EXTI_InitTypeDef EXTI_InitStructure;
	 EXTI_InitStructure.EXTI_Line = EXTI_Line13;
    EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
	 EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;  

    EXTI_InitStructure.EXTI_LineCmd = ENABLE;
    EXTI_Init(&EXTI_InitStructure);
	 
	 // Configure and enable EXTI line for magnetometer
	 EXTI_InitStructure.EXTI_Line = EXTI_Line11;
    EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
	 EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;  

    EXTI_InitStructure.EXTI_LineCmd = ENABLE;
    EXTI_Init(&EXTI_InitStructure);
	 
	 // Configure and enable EXTI line for accel
	 EXTI_InitStructure.EXTI_Line = EXTI_Line12;
    EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
	 EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;  

    EXTI_InitStructure.EXTI_LineCmd = ENABLE;
    EXTI_Init(&EXTI_InitStructure);	 
	
	 gSendStateData = 0;
	 
	 // Main program loop
    while(1)
    {
		  // If new data has arrived, use it to make new state estimates
		  if( gSensorData.new_gyro_data || gSensorData.new_accel_data || gSensorData.new_mag_data )
		  {
				EKF_EstimateStates( &gStateData, &gSensorData );
				gSensorData.new_gyro_data = 0;
				gSensorData.new_accel_data = 0;
				gSensorData.new_mag_data = 0;
		  }
		  
		  // If any of the sensors have new data, retrieve it
		  if( gGyroReady )
		  {
				gGyroReady = 0;
	  
				if ( getITGData( ) )
				{
					 gGyroStatesReceived = 1;
					 
					 gSensorData.gyro_x = -((g_i2cRxBuf[2] << 8) | g_i2cRxBuf[3]);
					 gSensorData.gyro_y = -((g_i2cRxBuf[0] << 8) | g_i2cRxBuf[1]);
					 gSensorData.gyro_z = -((g_i2cRxBuf[4] << 8) | g_i2cRxBuf[5]);
									 
					 if( gZeroGyroEnable )
					 {
						  gZeroGyroSampleCount++;
						  gZeroGyroAverages[0] += gSensorData.gyro_x;
						  gZeroGyroAverages[1] += gSensorData.gyro_y;
						  gZeroGyroAverages[2] += gSensorData.gyro_z;
						  
						  if( gZeroGyroSampleCount == GYRO_ZERO_SAMPLE_SIZE )
						  {
								StopGyroCalibration( );
								
								gGyrosCalibrated = 1;
						  }
					 }
					 
					 
					 gSensorData.new_gyro_data = 1;
				}
				else
				{
					 gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] |= UM6_I2C_GYRO_BUS_ERROR;
	//				SendStatusPacket();
				}

		  }
		  
		  if( gAccelReady )
		  {
				gAccelReady = 0;
				
				if( getLSMData( g_i2cRxBuf ) )
				{
					 gAccelStatesReceived = 1;
					 
					 gSensorData.accel_x = -((g_i2cRxBuf[1] << 8) | g_i2cRxBuf[0]);
					 gSensorData.accel_y = ((g_i2cRxBuf[3] << 8) | g_i2cRxBuf[2]);
					 gSensorData.accel_z = -((g_i2cRxBuf[5] << 8) | g_i2cRxBuf[4]);
									 
					 gSensorData.new_accel_data = 1;
				}
				else
				{
					 gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] |= UM6_I2C_ACCEL_BUS_ERROR;
//					 SendStatusPacket();
				}
				
		  }
		  
		  if( gMagReady )
		  {
				gMagReady = 0;
				
				// Get data.  If valid (return val = 1), then process and fill HMC data structure
				if( getHMCData() )
				{
					 gMagStatesReceived = 1;
					 
					 gSensorData.new_mag_data = 1;

					 gSensorData.mag_x = -((g_i2cRxBuf[0] << 8) | g_i2cRxBuf[1]);
					 gSensorData.mag_y = ((g_i2cRxBuf[2] << 8) | g_i2cRxBuf[3]);
					 gSensorData.mag_z = -((g_i2cRxBuf[4] << 8) | g_i2cRxBuf[5]);							
				}
				else
				{
					 gData.r[UM6_STATUS-DATA_REG_START_ADDRESS] |= UM6_I2C_MAG_BUS_ERROR;
//					 SendStatusPacket();
				}
				
		  }
		  
		  // Handle any new characters received by the UART (piped into memory via DMA)
		  HandleUSART1Reception();
	  
		  // Check if a packet has been received over the UART.  If so, respond.
		  if( gRXPacketReceived && !g_i2cBusy )
		  {
				ProcessPacket( (USARTPacket*)&gRXPacketBuffer[gRXPacketBufferStart] );

				// Increment RX packet buffer pointer.  The RX and TX buffer is circular.  The buffer is
				// empty if the "start" and "end" pointers point to the same location.
				gRXPacketBufferStart++;
				if( gRXPacketBufferStart >= RX_PACKET_BUFFER_SIZE )
				{
					 gRXPacketBufferStart = 0;
				}
				
				// If there are no more packets ready for processing, clear the gRXPacketReceived flag.
				// If there is another packet ready, leave the flag set so that the other RX packet will
				// be processed the next time through the main loop.
				if( gRXPacketBufferStart == gRXPacketBufferEnd )
				{
					 gRXPacketReceived = 0;
				}
		  }
		  
		  // If there is a packet waiting to be transmitted and the USART is inactive, then start transmission
		  if( gTXPacketBufferStart != gTXPacketBufferEnd)
		  {
				if( TXPacketBufferReady )
				{
					 TXPacketBufferReady = 0;
					 SendNextPacket( );
					 TXPacketBufferReady = 1;
				}						  
		  }
		  
		  // If a timer interrupt has occured signaling a need to transmit new state data, send the data
		  // (the global variable gSendStateData is set by a timer2 interrupt)
		  if( gSendStateData )
		  {
				gSendStateData = 0;
				SendDataPacket();
		  }
		  
		  // Refresh watchdog timer
//		  IWDG_ReloadCounter();
		  
    }
	 
}


/*******************************************************************************
* Function Name  : TIM2_IRQHandler
* Input          : None
* Output         : None
* Return         : None
* Description    : Called in response to a interrupt from Timer2.  Timer 2 is
						 used to trigger transmission of SENSOR_DATA packets periodically.
						 While the sampling sampling frequency of the ADC converter
						 remains fixed, data is transmitted at variable rates based
						 on the timer configuration.
*******************************************************************************/
void TIM2_IRQHandler( void )
{
	 if( TIM_GetITStatus(TIM2, TIM_IT_Update ) != RESET )
	 {
		  // If in broadcast mode and the gyros have been calibrated, transmit data packet
		  
		  if( (gConfig.r[UM6_COMMUNICATION] & UM6_BROADCAST_ENABLED) && gGyrosCalibrated )
		  {
				gSendStateData = 1;
		  }
		  
		  // Clear pending interrupt bit
		  TIM_ClearITPendingBit(TIM2, TIM_IT_Update);
	 }
}

/*******************************************************************************
* Function Name  : TIM4_IRQHandler
* Input          : None
* Output         : None
* Return         : None
* Description    : 

Called when Timer4 overflow occurs.  

This is used to gracefully recover from i2c bus errors.  When the timer overflows,
enough time has elapsed since the last overflow that data should have been received
by all sensors.  If data hasn't been received, then something is wrong.

This function causes requests to be sent to sensors that haven't communicated
new data to the processor since the last interrupt.

Mostly, this function is used to handle the magnetometer.  All the other sensors
fire interrupts every time new data is ready, but the magnetometer only sends
new interrupts when the old data has been read.  That means that if anything happens
(ie. a i2c bus error or a missed interrupt) then the magnetometer should be polled
so that it starts sending interrupts again.

*******************************************************************************/
void TIM4_IRQHandler( void )
{
	 static int flag = 0;
	 
	 if( TIM_GetITStatus(TIM4, TIM_IT_Update ) != RESET )
	 {		  
		  if( gMagStatesReceived == 0 )
		  {
				gMagReady = 1;
		  }
		  
		  if( gAccelStatesReceived == 0 )
		  {
				gAccelReady = 1;
		  }
		  
		  if( gGyroStatesReceived == 0 )
		  {
				gGyroReady = 1;
		  }
		  
		  gGyroStatesReceived = 0;
		  gAccelStatesReceived = 0;
		  gMagStatesReceived = 0;
		  	 
		  // Clear pending interrupt bit
		  TIM_ClearITPendingBit(TIM4, TIM_IT_Update);
	 }
}

/*******************************************************************************
* Function Name  : EXT15_10_IRQHandler
* Input          : None
* Output         : None
* Return         : None
* Description    : External interrupt handler for interupts 15 to 10.
*******************************************************************************/
void EXTI15_10_IRQHandler( void )
{	 
	 // Check for interrupt from ITG-3200 rate gyro
	 if( EXTI_GetITStatus(EXTI_Line13) != RESET )
	 {
		  gGyroReady = 1;
		  
		  EXTI_ClearITPendingBit(EXTI_Line13);
	 }
	 
	 // Check for interrupt from accels
	 if( EXTI_GetITStatus(EXTI_Line12) != RESET )
	 {
		  gAccelReady = 1;
		  
		  EXTI_ClearITPendingBit(EXTI_Line12);
	 }
	 
	 // Check for interrupt from magnetometer
	 if( EXTI_GetITStatus(EXTI_Line11) != RESET )
	 {		  
		  gMagReady = 1;

		  EXTI_ClearITPendingBit(EXTI_Line11);
	 }
	 
}

